<?php

namespace App\Http\Controllers;

use App\Events\JobApplied;
use App\Models\JobApplication;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class JobApplicationController extends Controller
{
    public function index(): JsonResponse
    {
        $jobApplications = JobApplication::with([
            'user' => function ($query) {
                $query->with(['candidateProfile', 'companyProfile']);
            },
            'job.company'
        ])->get();

        // Transform the data to include only the relevant profile based on User_type
        $jobApplications->each(function ($application) {
            if ($application->user->User_type === 'candidate') {
                $application->user->profile = $application->user->candidateProfile;
            } elseif ($application->user->User_type === 'company') {
                $application->user->profile = $application->user->companyProfile;
            }
            // Remove unnecessary profile relations to clean up the response
            unset($application->user->candidateProfile);
            unset($application->user->companyProfile);
        });

        return response()->json([
            'status' => 'success',
            'data' => $jobApplications,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $jobApplication = JobApplication::with([
            'user' => function ($query) {
                $query->with(['candidateProfile', 'companyProfile']);
            },
            'job.company'
        ])->findOrFail($id);

        // Include the relevant profile based on User_type
        if ($jobApplication->user->User_type === 'candidate') {
            $jobApplication->user->profile = $jobApplication->user->candidateProfile;
        } elseif ($jobApplication->user->User_type === 'company') {
            $jobApplication->user->profile = $jobApplication->user->companyProfile;
        }
        // Remove unnecessary profile relations
        unset($jobApplication->user->candidateProfile);
        unset($jobApplication->user->companyProfile);

        return response()->json([
            'status' => 'success',
            'data' => $jobApplication,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Job_id' => 'required|exists:jobs,Job_id',
            'Resume_url' => 'nullable|file|mimes:pdf,doc,docx|max:102400',
            'Cover_letter' => 'nullable|string',
            'Status' => 'sometimes|in:draft,pending,reviewing,shortlisted,interviewing,accepted,rejected',
            'Rating' => 'nullable|numeric|between:0,10',
            'Screening_score' => 'nullable|numeric|between:0,99.99',
            'Overall_score' => 'nullable|numeric|between:0,99.99',
        ], [
            'Resume_url.file' => 'The resume must be a valid file.',
            'Resume_url.mimes' => 'The resume must be a PDF, DOC, or DOCX file.',
            'Resume_url.max' => 'The resume must not exceed 100MB.',
            'Screening_score.between' => 'The screening score must be between 0 and 99.99.',
            'Overall_score.between' => 'The overall score must be between 0 and 99.99.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'Job_id',
            'Cover_letter',
            'Status',
            'Rating',
            'Screening_score',
            'Overall_score',
        ]);

        if ($request->hasFile('Resume_url') && $request->file('Resume_url')->isValid()) {
            $resume = $request->file('Resume_url');
            $resumeName = time() . '_' . $resume->getClientOriginalName();
            $resumePath = public_path('storage/job_applications/resumes');
            $resume->move($resumePath, $resumeName);
            $data['Resume_url'] = asset('storage/job_applications/resumes/' . $resumeName);
        } elseif ($request->has('Resume_url')) {
            $contentInput = $request->input('Resume_url');
            if (is_string($contentInput) && filter_var($contentInput, FILTER_VALIDATE_URL) && strpos($contentInput, asset('storage/job_applications/resumes/')) === 0) {
                $data['Resume_url'] = $contentInput;
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Resume must be a valid storage URL',
                ], 422);
            }
        }

        $jobApplication = JobApplication::create($data);

        // Fire the JobApplied event
        try {
            event(new JobApplied($jobApplication));
            Log::info('JobApplied event fired for job application', [
                'application_id' => $jobApplication->Application_id,
                'job_id' => $jobApplication->Job_id,
                'user_id' => $jobApplication->User_id,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to fire JobApplied event: ' . $e->getMessage(), [
                'application_id' => $jobApplication->Application_id,
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => $jobApplication,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $jobApplication = JobApplication::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'Job_id' => 'sometimes|exists:jobs,Job_id',
            'Resume_url' => 'sometimes|nullable|string',
            'Cover_letter' => 'nullable|string',
            'Status' => 'sometimes|in:draft,pending,reviewing,shortlisted,interviewing,accepted,rejected',
            'Rating' => 'nullable|numeric|between:0,10',
            'Screening_score' => 'nullable|numeric|between:0,99.99',
            'Overall_score' => 'nullable|numeric|between:0,99.99',
        ], [
            'Screening_score.between' => 'The screening score must be between 0 and 99.99.',
            'Overall_score.between' => 'The overall score must be between 0 and 99.99.',
        ]);

        if ($validator->fails()) {
            Log::warning("Validation failed for job application update", [
                'job_application_id' => $id,
                'errors' => $validator->errors()->toArray()
            ]);
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'Job_id',
            'Cover_letter',
            'Status',
            'Rating',
            'Screening_score',
            'Overall_score',
        ]);

        $validMimes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];

        if ($request->has('Resume_url') && !empty($request->input('Resume_url'))) {
            $contentInput = $request->input('Resume_url');
            if ($this->isValidBase64($contentInput)) {
                $mimeType = $this->getMimeTypeFromBase64($contentInput);
                if (!in_array($mimeType, $validMimes)) {
                    return response()->json([
                        'status' => 'error',
                        'message' => "Invalid MIME type: $mimeType, expected: " . implode(', ', $validMimes),
                    ], 422);
                }

                $base64String = preg_replace('/^data:application\/[\w\.-]+;base64,/', '', $contentInput);
                $contentData = base64_decode($base64String, true);
                if ($contentData === false) {
                    Log::warning("Failed to decode base64 string for Resume_url in job application update", [
                        'job_application_id' => $id,
                        'content_input' => substr($contentInput, 0, 100) . '...'
                    ]);
                    $data['Resume_url'] = $jobApplication->Resume_url;
                } elseif (strlen($contentData) > 102400 * 1024) {
                    Log::warning("Resume_url file size exceeds 100MB in job application update", [
                        'job_application_id' => $id,
                        'size' => strlen($contentData)
                    ]);
                    $data['Resume_url'] = $jobApplication->Resume_url;
                } else {
                    if ($jobApplication->Resume_url) {
                        $oldContentPath = str_replace(asset('storage/'), public_path('storage/'), $jobApplication->Resume_url);
                        if (File::exists($oldContentPath)) {
                            File::delete($oldContentPath);
                            Log::info("Deleted old resume file for job application", [
                                'job_application_id' => $id,
                                'old_content_path' => $oldContentPath
                            ]);
                        }
                    }

                    $contentExtension = [
                        'application/pdf' => 'pdf',
                        'application/msword' => 'doc',
                        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx'
                    ][$mimeType];
                    $contentName = time() . '_resume.' . $contentExtension;
                    $contentPath = public_path('storage/job_applications/resumes/' . $contentName);
                    File::put($contentPath, $contentData);
                    $data['Resume_url'] = asset('storage/job_applications/resumes/' . $contentName);
                    Log::info("Successfully updated Resume_url with base64 for job application", [
                        'job_application_id' => $id,
                        'new_content_path' => $contentPath
                    ]);
                }
            } elseif (filter_var($contentInput, FILTER_VALIDATE_URL) && strpos($contentInput, asset('storage/job_applications/resumes/')) === 0) {
                if ($jobApplication->Resume_url && $jobApplication->Resume_url !== $contentInput) {
                    $oldContentPath = str_replace(asset('storage/'), public_path('storage/'), $jobApplication->Resume_url);
                    if (File::exists($oldContentPath)) {
                        File::delete($oldContentPath);
                        Log::info("Deleted old resume file for job application due to new URL", [
                            'job_application_id' => $id,
                            'old_content_path' => $oldContentPath
                        ]);
                    }
                }
                $data['Resume_url'] = $contentInput;
                Log::info("Updated Resume_url with valid storage URL for job application", [
                    'job_application_id' => $id,
                    'resume_url' => $contentInput
                ]);
            } else {
                Log::warning("Invalid Resume_url provided in job application update", [
                    'job_application_id' => $id,
                    'content_input' => substr($contentInput, 0, 100) . '...'
                ]);
                $data['Resume_url'] = $jobApplication->Resume_url;
            }
        } elseif ($request->hasFile('Resume_url') && $request->file('Resume_url')->isValid()) {
            if ($jobApplication->Resume_url) {
                $oldContentPath = str_replace(asset('storage/'), public_path('storage/'), $jobApplication->Resume_url);
                if (File::exists($oldContentPath)) {
                    File::delete($oldContentPath);
                    Log::info("Deleted old resume file for job application due to new file upload", [
                        'job_application_id' => $id,
                        'old_content_path' => $oldContentPath
                    ]);
                }
            }

            $resume = $request->file('Resume_url');
            $fileMime = $resume->getMimeType();
            if (!in_array($fileMime, $validMimes)) {
                return response()->json([
                    'status' => 'error',
                    'message' => "Invalid MIME type: $fileMime, expected: " . implode(', ', $validMimes),
                ], 422);
            }
            $resumeName = time() . '_' . $resume->getClientOriginalName();
            $resumePath = public_path('storage/job_applications/resumes');
            $resume->move($resumePath, $resumeName);
            $data['Resume_url'] = asset('storage/job_applications/resumes/' . $resumeName);
            Log::info("Successfully updated Resume_url with file upload for job application", [
                'job_application_id' => $id,
                'new_content_path' => $resumePath . '/' . $resumeName
            ]);
        }

        $jobApplication->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $jobApplication,
            'message' => isset($contentInput) && !empty($contentInput) && $data['Resume_url'] === $jobApplication->Resume_url
                ? 'Job application updated, invalid resume URL or base64 string was skipped'
                : 'Job application updated successfully',
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $jobApplication = JobApplication::findOrFail($id);
        if ($jobApplication->Resume_url) {
            $contentPath = str_replace(asset('storage/'), public_path('storage/'), $jobApplication->Resume_url);
            if (File::exists($contentPath)) {
                File::delete($contentPath);
            }
        }
        $jobApplication->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Job application deleted successfully',
        ], 200);
    }

    /**
     * Helper method to get MIME type from base64 string
     */
    private function getMimeTypeFromBase64($base64String)
    {
        if (preg_match('/^data:application\/([\w\.-]+);base64,/', $base64String, $matches)) {
            return 'application/' . $matches[1];
        }

        $base64Data = preg_replace('/^data:application\/[\w\.-]+;base64,/', '', $base64String);
        $decodedData = base64_decode($base64Data, true);

        if ($decodedData === false) {
            return 'invalid';
        }

        if (substr($decodedData, 0, 4) === '%PDF') {
            return 'application/pdf';
        }

        return 'application/octet-stream';
    }

    /**
     * Helper method to validate base64 string
     */
    private function isValidBase64($string)
    {
        $base64Data = preg_replace('/^data:application\/[\w\.-]+;base64,/', '', $string);
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $base64Data)) {
            return false;
        }
        if (strlen($base64Data) % 4 !== 0) {
            return false;
        }
        return true;
    }
}
